//
//  PrintfESCManager.h
//  TestPrint
//
//  Created by Mac on 5/8/21.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIImage.h>
#import "ESCTable.h"
NS_ASSUME_NONNULL_BEGIN
typedef NS_ENUM(NSInteger, BarCodeType) {
    BarCodeTypeUPCA = 0,
    BarCodeTypeUPCE = 1,
    BarCodeTypeEAN13 = 2,
    BarCodeTypeEAN8 = 3,
    BarCodeTypeCODE39 = 4,
    BarCodeTypeITF = 5,
    BarCodeTypeCODABAR = 6,
    BarCodeTypeCODE93 = 7,
    BarCodeTypeCODE128 = 8
};

typedef NS_ENUM(NSInteger, BarCodePositionType){
    BarCodePositionTypeNONE = 0,//不打印 Don't print
    BarCodePositionTypeABOVE,//条码上方 Above the barcode
    BarCodePositionTypeBELOW,//条码下方 Below the barcode
    BarCodePositionTypeBOTH //条码上下方都有 There are both above and below the barcode
};

typedef NS_ENUM(NSInteger, ALIGN) {
    ALIGNLETF = 0,//居左 On the left
    ALIGNCENTER,//居中 Centered
    ALIGNRIGHT//居右 Right
};

@interface PrintfESCManager : NSObject


+(instancetype)createNew;

/**
 * ESC @ Initialize the printer
 * Clear the print buffer data, the print mode is set to the default mode at power-on
 */
-(void)initPrinter;

/**
 * Set GB18030 encoding format text
 */
-(void)setText:(NSString*)content;

/**
 *LF print and wrap
 *Print out the data in the print buffer, and advance the printing paper one line according to the current line spacing.
 */
-(void)setPrint;

/**
 * ESC d print and feed paper n lines
 * Print the data in the buffer and feed the paper n lines (character lines)
 */
-(void)setPrintAndFeedRow:(int)row;

/**
 * ESC $ set absolute printing position
 * Set the current position to the distance from the beginning of the line (nL + nH × 256) × (horizontal or vertical movement unit).
 * The incoming parameter is the number of points
 * 1mm=8dot
 */
-(void)setAbsolutePrintPosition:(int)where;

/**
 * ESC a 选择对齐方式
 * 使所有的打印数据按某一指定对齐方式排列。
 * n 的取值与对齐方式对应关系如下
 * • 当n 为0 时 ： 左对齐
 * • 当n 为1 时 ： 中间对齐
 * • 当n 为2 时  右对齐
 */

/**
  * ESC a to select alignment
  * Make all the print data arranged according to a specified alignment.
  * The corresponding relationship between the value of n and the alignment is as follows
  * • When n is 0: left-justified
  * • When n is 1: aligned in the middle
  * • When n is 2, align right
  */
-(void)setSelectJustification:(int)which;

/**
 * GS L 设置左边距
 * 传入参数为点数
 * 1mm=8dot
 */
/**
  * GS L sets the left margin
  * The incoming parameter is the number of points
  * 1mm=8dot
  */
-(void)setLeftMargin:(int)left;

/**
 * ESC 2 设置默认行间距
 *选择默认行间距
 */
/**
  * ESC 2 sets the default line spacing
  *Select the default line spacing
  */
-(void)setDefaultLineSpace;

/**
 * ESC 3 设置行间距
 * 传入参数为点数
 * 1mm=8dot
 */

/**
  * ESC 3 set line spacing
  * The incoming parameter is the number of points
  * 1mm=8dot
  */
-(void)setLineSpace:(int)space;

/**
 * ESC ! 选择打印模式
 * 传入参数说明
 * 根据n 的值设置字符打印模式
 *  0：标准ASCII码字体 (12 × 24)
 *  1：压缩ASCII码字体B (9 × 17)
 *  16：选择倍高模式
 *  32：选择倍宽模式
 *
 *  多个模式，相加即可
 */

/**
  * ESC! Select print mode
  * Incoming parameter description
  * Set the character printing mode according to the value of n
  * 0: Standard ASCII code font (12 × 24)
  * 1: Compressed ASCII code font B (9 × 17)
  * 16: Select double height mode
  * 32: select double width mode
  *
  * Multiple modes, just add them together
  */
-(void)setPrintMode:(int)mode;

/**
 * ESC – 选择/取消下划线模式
 * 传入参数说明
 * • 当n 为0 时：取消下划线模式
 * • 当n 为1 时：选择下划线模式（1 点宽）
 * • 当n 为2 时：选择下划线模式（2 点宽）
 */

/**
  * ESC-select/cancel underline mode
  * Incoming parameter description
  * • When n is 0: cancel the underline mode
  * • When n is 1: select the underline mode (1 point wide)
  * • When n is 2: select underline mode (2 points wide)
  */
-(void)setUnderlineMode:(int)mode;

/**
 * ESC E 选择/取消加粗模式
 * 传入参数说明
 * 当n 为0 时，取消加粗模式。
 * 当n 为1 时，选择加粗模式。
 */

/**
  * ESC E select/cancel bold mode
  * Incoming parameter description
  * When n is 0, the bold mode is cancelled.
  * When n is 1, select bold mode.
  */
-(void)setBoldMode:(int)mode;


/**
 * ESC M 选择字体
 * 传入参数说明
 * • 当n 位为0 时， 选择标准ASCII 码字体(12 × 24)
 * • 当n 位为1 时， 选择压缩ASCII 码字体(9 × 17))
 */

/**
  * ESC M to select font
  * Incoming parameter description
  * • When the n bit is 0, select the standard ASCII code font (12 × 24)
  * • When the n bit is 1, select the compressed ASCII code font (9 × 17))
  */
-(void)setSelectFont:(int)font;


/**
 * 打印条码
 * 传入参数说明
 * type：条码类型。0：UPC-A，1：UPC-B，2：EAN13，3：EAN8，4：CODE39，5：ITF，6：CODABAR，7：CODE93，8：CODE128
 * barcodeStringPosition：条码字符打印位置。0：不打印，1：条码上方，2：条码下方，3：条码上、下方都打印，
 * barcodeHeightInDot：条码高度。高度范围1 ≤ n ≤ 255。
 * barcodeWidth：条码宽度。条码宽度范围2 ≤ n ≤ 6。
 */
/**
  * Print barcode
  * Incoming parameter description
  * type: barcode type. 0: UPC-A, 1: UPC-B, 2: EAN13, 3: EAN8, 4: CODE39, 5: ITF, 6: CODABAR, 7: CODE93, 8: CODE128
  * barcodeStringPosition: barcode character printing position. 0: Do not print, 1: Above the barcode, 2: Below the barcode, 3: Print both above and below the barcode,
  * barcodeHeightInDot: barcode height. Height range 1 ≤ n ≤ 255.
  * barcodeWidth: barcode width. Bar code width range 2 ≤ n ≤ 6.
  */
-(void)setBarCodeWithType:(int)type barcodeStringPosition:(int)position barcodeHeightInDot:(int)height barcodeWidth:(int)width barcodeContent:(NSString*)content;

/**
 *打印二维码
 *传入参数说明
 *content：二维码内容。
 *size: 二维码大小。0 ≤ n ≤ 3。
 */
/**
 * Print the QR code
 * Incoming parameter description
 * content: QR code content.
 * size: The size of the QR code. 0 ≤ n ≤ 3.
 * Width: Bar code width. Bar code width range 2 ≤ n ≤ 6.
 */
-(void)setQRCode:(NSString *)content size:(int)size;

/**
 * 打印二维码
 * 传入参数说明
 * content：二维码内容。
 */
/**
 * Print QR code
 * Incoming parameter description
 * content: QR code content.
 */
-(void)setQRCode:(NSString *)content;

/**
 * 打满虚线,58mm
 * Full dotted line, 58mm
 */
-(void)setPlusLine_58;

/**
 * 打满虚线，80mm
 * Full dotted line, 80mm
 */
-(void)setPlusLine_80;

/**
 * 打印图片
 * 传入参数说明
 * image：图片，宽度不要超过打印机的有效打印宽度，比如58mm打印机，有效打印宽度为48mm，那么图片的宽度最大48*8px
 * left：距离左边的距离
 */

/**
 * Print pictures
 * Incoming parameter description
 * image: The width of the picture should not exceed the effective printing width of the printer. For example, for a 58mm printer, the effective printing width is 48mm, then the maximum width of the picture is 48*8px
 * left: the distance from the left
 */
-(void)setImage:(UIImage *)image left:(int)left;


/**
 * 打印标题和内容，分两边
 * 传入参数说明
 * title：标题，即左边的文字
 * content：内容，即右边的文字
 * paperMM：纸张大小，58mm打印机传48，80mm打印机传72
 */
/**
 * Print title and content, divided into two sides
 * Incoming parameter description
 * title: title, which is the text on the left
 * content: content, that is, the text on the right
 * paperMM: paper size, 58mm printer passes 48, 80mm printer passes 72
 */
-(void)setTwoColumn:(NSString *)title content:(NSString *)content paperMM:(int)paperMM;

/**
 * 打印表格
 * 传入参数说明
 * table：ESCTable对象，关于ESCTable对象的使用，请看其头文件
 */
/**
 * Print form
 * Incoming parameter description
 * table: ESCTable object, for the use of ESCTable object, please see its header file
 */
-(void)setTable:(ESCTable*)table;


/**
 * 获取打印数据
 * Get print data
 */
-(NSData *)getData;

/**
 * 清空打印数据
 * Clear print data
 */
-(void)clearData;

@end

NS_ASSUME_NONNULL_END
